<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
require_once __DIR__ . '/BotInterface.php';

class OpenRouterBot implements BotInterface {
    public static function getKey() {
        return 'openrouter';
    }
    
    public static function getName() {
        return 'OpenRouter';
    }
    
    public static function getTemplate() {
        $models = self::getModels();
        return array(
            "openrouter_token"  => array( "type" => "text",     "default"=>""),
            "model"             => array( "type" => "select",   "default"=>"qwen/qwen3-14b:free", "options"=>$models),
            "system_prompt"     => array( "type" => "textarea", "default"=>"You are a helpful assistant"),
            "action_prompt"     => array( "type" => "textarea", "default"=>"If the user is frustrated or angry, or if the user asks to connect to a real person, you will pass the conversation to a human by appending '[transfer]' to your reply. In the farewell/goodbye message you will append '[close]' to your reply."),
            "guardrails_prompt" => array( "type" => "textarea", "default"=>". Do not answer anything that is not in your system prompt context."),
            "debug"             => array( "type" => "select",   "default"=>"no", "options"=>array("yes"=>__('Yes'),"no"=>__('No')))
        );
    }

    public static function getModels() {
        $cacheFile = __DIR__ . '/openrouter_models.json';
        $cacheDuration = 60 * 60 * 24 * 7; // 1 week
    
        if (file_exists($cacheFile) && 
            (time() - filemtime($cacheFile)) < $cacheDuration) {
            $cachedData = file_get_contents($cacheFile);
            $models = json_decode($cachedData, true);
        } else {
            $ch = curl_init('https://openrouter.ai/api/v1/models');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            $response = curl_exec($ch);
        
            if (curl_errno($ch)) {
                if (file_exists($cacheFile)) {
                    $models = json_decode(file_get_contents($cacheFile), true);
                } else {
                    $models = array();
                }
            } else {
                $data = json_decode($response, true);
                $models = array();
                if (!empty($data['data'])) {
                    foreach ($data['data'] as $model) {
                        $models[$model['id']] = $model['id'];
                    }
                    file_put_contents($cacheFile, json_encode($models));
                }
            }
        }
    
        // Sort models with free ones first, then alphabetically
        uksort($models, function($a, $b) {
            $aFree = (strpos($a, ':free') !== false);
            $bFree = (strpos($b, ':free') !== false);
            // If both are free or both are not free, sort alphabetically
            if ($aFree === $bFree) {
                return strcasecmp($a, $b);
            }
            // Put free models first
            return $aFree ? -1 : 1;
        });
        return $models;
    }



}

if (!defined('BOT_REGISTRATION_MODE') || !BOT_REGISTRATION_MODE) {
    require_once __DIR__ . '/common_functions.php';

    $iniFile = isset($_REQUEST['bot']) ? getcwd()."/".$_REQUEST['bot'].".ini" : null;
    $config = BotConfig::load('OpenRouterBot', $iniFile);
    $logfile = basename(__FILE__,'.php');

    if ($config === null) {
        header('HTTP/1.0 403 Forbidden');
        die("Configuration file missing or inaccessible");
    }

    // Get bot input DATA
    $json = file_get_contents('php://input');
    $data = json_decode($json,1);

    if(!isset($data)) { 
        header('HTTP/1.0 403 Forbidden');
        die("Bad call");
    }
    bot_debug($logfile,"POST RECEIVED:\n".print_r($data,1));

    $user_prompt = $data['body'];

    $user_prompt.=$config['guardrails_prompt']; // add guardrails to question

    $chat_context = array_slice($data['chat_context'],-5);
    $chat_context[] = array("role"=>"user","content"=>$user_prompt);

    if($config['openrouter_token']=='') {
        $response = "Bot is not yet configured, be sure to set up your OpenRouter API key";
    } else {
        $response = ask_openrouter($config['openrouter_token'],$config['system_prompt']." ".$config['action_prompt'],$chat_context);
    }

    $return = array("body"=>$response,"attach"=>'');
    bot_debug($logfile,"REPLY: $response");
    echo json_encode($return);
    exit;
}

function ask_openrouter($token,$system_prompt,$user_prompt) {
    global $config, $logfile;

    $url = 'https://openrouter.ai/api/v1/chat/completions';
    $sysprompt = array();
    $sysprompt[]=array("role"=>"system","content"=>$system_prompt);
    $messages = array_merge($sysprompt,$user_prompt);
    $data = array(
       "model"=> $config['model'],
       "messages" => $messages
    );
    $data_string = json_encode($data);
    bot_debug($logfile,"OpenRouter Messages: ".print_r($messages,1));
    $headers=array('Content-Type: application/json','Content-Length: ' . strlen($data_string),"Authorization: Bearer $token");
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_HEADER, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120);
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    curl_setopt($ch, CURLOPT_SSLVERSION, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $return = curl_exec($ch);
    curl_close($ch);
    $response = json_decode($return,1);
    bot_debug($logfile,"OpenRouter response: ".print_r($response,1));
    return $response['choices'][0]['message']['content'];
}
